/*******************************************************************************
* manager.cpp: Vls manager
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: manager.cpp,v 1.7 2002/03/25 01:23:54 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include "config.h"

#include "../core/core.h"

#include "../mpeg/mpeg.h"
#include "../mpeg/ts.h"

#include "program.h"
#include "request.h"
#include "admin.h"
#include "buffer.h"
#include "output.h"
#include "channel.h"
#include "broadcast.h"
#include "input.h"
#include "repository.h"
#include "directory.h"
#include "tsstreamer.h"

#include "manager.h"
#include "vls.h"

#include "repository.cpp"


/*******************************************************************************
* C_Manager
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Manager::C_Manager(handle hLog, C_EventHub* pHub) : m_cEventQueued(0)
{
  ASSERT(hLog);

  m_hLog = hLog;
  m_pEventHub = pHub;

  m_bStop = false;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_Manager::~C_Manager()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_Manager::Init()
{
  int iRc = NO_ERR;

  try
  {
    InitChannels();
    InitInputs();
    InitPgrmTable();
    Create();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to init manager: " + e.Dump());
    iRc = GEN_ERR;
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_Manager::Run()
{
  int iRc = NO_ERR;

  // Nothing to do yet

  return iRc;
}


//------------------------------------------------------------------------------
// Stop the execution
//------------------------------------------------------------------------------
int C_Manager::Stop()
{
  Log(m_hLog, LOG_NOTE, "Stopping the manager");

  int iRc = NO_ERR;

  try
  {
    StopPrograms();
    C_Thread::Stop();
    Log(m_hLog, LOG_NOTE, "Manager stopped");
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to stop the manager:\n" + e.Dump());
    iRc = GEN_ERR;
  } 

  return iRc;
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_Manager::Destroy()
{
  int iRc = NO_ERR;

  Log(m_hLog, LOG_NOTE, "Destroying the manager");

  try
  {
    DestroyPgrmTable();
    DestroyInputs();
    DestroyChannels();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to destroy manager: " + e.Dump());
    iRc = e.GetCode();
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Manager destroyed");

  return iRc;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::InitPgrmTable()
{
  // Lock the input repository and the program list
  m_cInputList.Lock();
  m_cProgramList.Lock();

  // Get the input lists of programs and merge them with the manager one
  C_RepositoryBrowser<C_String, C_Input> cIterator =
                                                  m_cInputList.CreateBrowser();
  
  while(cIterator.HasNextItem())
  {
    C_Input* pInput = cIterator.GetNextItem();
    C_List<C_Program> cInputPgrms = pInput->GetAvailablePgrms();
    unsigned int iPgrmNumber = cInputPgrms.Size();
    for(unsigned int j = 0; j < iPgrmNumber; j++)
    {
      m_cProgramList.Add(cInputPgrms[j], pInput);
    }
  }
  
  // Unlock the input repository and the program list
  m_cInputList.UnLock();
  m_cProgramList.UnLock();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::DestroyPgrmTable()
{
  // Nothing to do yet
}



//------------------------------------------------------------------------------
// Inputs initialization
//------------------------------------------------------------------------------
void C_Manager::InitInputs()
{
  // Get the names of all the sources
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  C_Vector<C_Setting> vInputs = pApp->GetSettings("Inputs");

  // Lock the input repository
  m_cInputList.Lock();

  // Create the corresponding inputs
  for(unsigned int i = 0; i < vInputs.Size(); i++)
  {
    // Current input
    C_Input* pInput = NULL;

    // Get input name and type
    C_Setting cCurrentInput = vInputs[i];
    C_String strInputName = cCurrentInput.GetName();
    C_String strInputType = cCurrentInput.GetValue();

    // Create the input for the given source
    C_InputModule* pModule = (C_InputModule*)
                             C_Application::GetModuleManager()
                                        ->GetModule("input", strInputType);
    if(pModule)
    {
      pInput = pModule->NewInput(strInputName);
      ASSERT(pInput);
      pInput->SetEventHandler(this);
      Log(m_hLog, LOG_NOTE, "Starting input '"+strInputName+"'");
      try
      {
        // Initialize the input
        pInput->Init();

        // And register it
        m_cInputList.Add(strInputName, pInput);

        Log(m_hLog, LOG_NOTE, "Input '" + strInputName +
            "' sucessfully initialised");
      }
      catch(E_Exception e)
      {
        Log(m_hLog, LOG_ERROR, "Unable to start input '" + strInputName +
            "': " + e.Dump());
        delete pInput;
      }
    }
    else
    {
      Log(m_hLog, LOG_ERROR, "Input type \"" + strInputType + "\" invalid");
    }
  }

  // Unlock the input repository
  m_cInputList.UnLock();
}


//------------------------------------------------------------------------------
// Inputs destruction
//------------------------------------------------------------------------------
void C_Manager::DestroyInputs()
{
  // Lock the input repository
  m_cInputList.Lock();

  C_RepositoryBrowser<C_String, C_Input> cIterator =
                                                m_cInputList.CreateBrowser();
  
  while(cIterator.HasNextItem())
  {
    C_Input* pInput = cIterator.GetNextItem();
    
    try
    {
      pInput->Destroy();
      Log(m_hLog, LOG_NOTE, "Input "+pInput->GetName()+" correctly stopped");
    }
    catch(E_Exception e)
    {
      // Just log the pb and go on with inputs destruction
      C_String strInputName = pInput->GetName();
      Log(m_hLog, LOG_ERROR,
          "Unable to stop input "+strInputName+": "+e.Dump());
    }
  }

  // Unlock the input repository
  m_cInputList.UnLock();
}


//------------------------------------------------------------------------------
// Channels initialization
//------------------------------------------------------------------------------
void C_Manager::InitChannels()
{
  // Lock the channel repository
  m_cChannelList.Lock();

  // Get the names of all the channels
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  C_Vector<C_Setting> vChannels = pApp->GetSettings("Channels");
  
  // Create the corresponding channels
  for(unsigned int i = 0; i < vChannels.Size(); i++)
  {
    // Current channel
    C_Channel* pChannel = NULL;

    // Get channel name and type
    C_Setting cCurrentChannel = vChannels[i];
    C_String strChannelName = cCurrentChannel.GetName();
    C_String strChannelType = cCurrentChannel.GetValue();

    C_ChannelModule* pModule = (C_ChannelModule*)
                               C_Application::GetModuleManager()
                                        ->GetModule("channel", strChannelType);
    if(pModule)
    {
      pChannel = pModule->NewChannel(strChannelName);
      ASSERT(pChannel);
      m_cChannelList.Add(strChannelName, pChannel);
      Log(m_hLog, LOG_NOTE, "Channel '"+strChannelName+"' created");
    }
    else
    {
      Log(m_hLog, LOG_ERROR, "Channel type \"" + strChannelType + "\" invalid");
    }
  }

  // Unlock the channel repository
  m_cChannelList.UnLock();
}


//------------------------------------------------------------------------------
// Channels destruction
//------------------------------------------------------------------------------
void C_Manager::DestroyChannels()
{
  // Just make sure that the channel is free
  // to do
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::StopPrograms()
{
  // Stop all th programs so that the inputs could be destroyed

  m_cBroadcasts.Lock();

  C_Vector<C_Request> cRequests;

  C_RepositoryBrowser<C_String, C_Broadcast> cIterator1 =
                m_cBroadcasts.CreateBrowser();

  while(cIterator1.HasNextItem())
  {
    C_Broadcast* pBroadcast = cIterator1.GetNextItem();
    C_Request* pRequest = new C_Request("stop");
    pRequest->SetArg("input", pBroadcast->GetInput()->GetName());
    pRequest->SetArg("program", pBroadcast->GetProgram()->GetName());
    cRequests.Add(pRequest);
  }

  m_cBroadcasts.UnLock();

  unsigned int iCount = cRequests.Size();
  for(unsigned int i = 0; i < iCount; i++)
  {
    C_Request& cRequest = cRequests[i];
    HandleRequest(cRequest);
    LogDbg(m_hLog, "Remove the broadcast");
    m_cBroadcasts.Lock();
    int iRc = m_cBroadcasts.Remove(cRequest.GetArg("program") + ":" +
                                   cRequest.GetArg("input"));
    m_cBroadcasts.UnLock();
    LogDbg(m_hLog, C_String("Broadcast removed with status ") + iRc);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::HandleRequest(const C_Request& cRequest)
{
  m_cCmdLock.Lock();
  LogDbg(m_hLog, "Manager locked");

  C_Answer cAnswer("Manager");
  
  C_String strCmd = cRequest.GetCmd();

  if(strCmd == "load")
  {
    // Load the input: Not yet implemented 
    ASSERT(false);
  }
  else if(strCmd == "unload")
 {
    // Unload the input: Not yet implemented 
    ASSERT(false);
  }
  else if(strCmd == "browse")
  {
    cAnswer = Browse(cRequest);
  }
  else if(strCmd == "start")
  {
    cAnswer = Start(cRequest);
  }
  else if(strCmd == "resume")
  {
    cAnswer = Resume(cRequest);
  }
  else if(strCmd == "suspend")
  {
    cAnswer = Suspend(cRequest);
  }
  else if(strCmd == "stop")
  {
    cAnswer = Stop(cRequest);
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown command: "+strCmd);
  }

  m_cCmdLock.UnLock();
  LogDbg(m_hLog, "Manager unlocked");

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::HandleEvent(const C_Event& cEvent)
{
  m_cCmdLock.Lock();
  LogDbg(m_hLog, "Manager locked");

  m_cEventFifoLock.Lock();
  m_cEventFifo.Add(new C_Event(cEvent));
  m_cEventFifoLock.UnLock();

  m_cEventQueued.Post();

  m_pEventHub->ForwardEvent(cEvent);

  m_cCmdLock.UnLock();
  LogDbg(m_hLog, "Manager unlocked");
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::PrivHandleEvent(const C_Event& cEvent)
{
  switch(cEvent.GetCode())
  {
  case EOF_EVENT :
  case CONNEXION_LOST_EVENT :
    C_String strInput = cEvent.GetBroadcast()->GetInput()->GetName();
    C_String strPgrm = cEvent.GetBroadcast()->GetProgram()->GetName();

    ASSERT(strInput != "");
    ASSERT(strPgrm != "");

    C_Request cRequest("stop");
    cRequest.SetArg("input", strInput);
    cRequest.SetArg("program", strPgrm);

    C_Answer cAnswer = HandleRequest(cRequest);

    break;
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::InitWork()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::DoWork()
{
  while(!m_bStop)
  {
    m_cEventQueued.Wait();

    m_cEventFifoLock.Lock();

    if((m_cEventFifo.Size() >= 1) && (!m_bStop))
    {
      C_Event* pEvent = m_cEventFifo.Remove(0);
      ASSERT(pEvent);

      m_cEventFifoLock.UnLock();

      PrivHandleEvent(*pEvent);

      delete pEvent;
    }
    else
    {
      ASSERT(m_bStop);

      m_cEventFifoLock.UnLock();
    }
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::StopWork()
{
   m_bStop = true;
   m_cEventQueued.Post();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::CleanWork()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Browse(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);
  cAnswer.AddMessage("Program Table");

  m_cProgramList.Lock();

  C_PgrmDirectoryBrowser cBrowser = m_cProgramList.CreateBrowser();
  
  while(cBrowser.HasNext())
  {
    C_PgrmInfo* pPgrmInfo = cBrowser.GetNext();
    ASSERT(pPgrmInfo);

    // Build a subanswer with the pgrm description and the list of providers
    const C_Program* pPgrm = pPgrmInfo->GetProgram();
    C_String strPgrmName = pPgrm->GetName();
    ASSERT(strPgrmName != "");
    C_Answer cPgrmDescr(strPgrmName);
    cPgrmDescr.SetStatus(NO_ERR);
    cPgrmDescr.AddMessage(pPgrm->GetDescription());

    const C_Vector<C_Input>& cProviders = pPgrmInfo->GetProviders();
    C_String strProviders = "Provided by";
    for(unsigned int i = 0; i < cProviders.Size(); i++)
    {
      strProviders += " " + cProviders[i].GetName();
    }
    cPgrmDescr.AddMessage(strProviders);

    // Add it to the answer
    cAnswer.Add(cPgrmDescr);
  }

  m_cProgramList.UnLock();

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Start(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  // Find the channel
  C_String strChannel = cRequest.GetArg("channel");
  ASSERT(strChannel != "");
  C_Channel* pChannel = m_cChannelList.Get(strChannel);
  if(!pChannel)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown channel");
  }
  else if(!pChannel->IsFree())
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Channel is busy");
    pChannel = NULL;
  }

  // Find the pgrm
  C_String strPgrm = cRequest.GetArg("program");
  ASSERT(strPgrm != "");
  const C_Program* pPgrm = m_cProgramList.GetPgrm(strPgrm);
  if(!pPgrm)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown pgrm");
  }  
   
  // Find the input
  C_String strInput = cRequest.GetArg("input");
//  if(strInput == "")
//  {
//    strInput = m_cPgrmList.GetInput(strPgrm);
//  }
//  ASSERT(strInput != "");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
  }

  // kludge; Dkludg ? --Bozo
  if(pPgrm && pInput && pChannel)
  {
    C_Broadcast* pBroadcast = new C_Broadcast(pPgrm, pInput, pChannel);
    pBroadcast->SetOptions(cRequest.GetArgs());

    C_Answer cInputAnswer = pInput->StartStreaming(pBroadcast);

    if(!cInputAnswer.GetStatus())
    {
      m_cBroadcasts.Lock();
      m_cBroadcasts.Add(strPgrm + ":" + strInput, pBroadcast);
      m_cBroadcasts.UnLock();
    }
    else
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
    }

    cAnswer.Add(cInputAnswer);
  }
  else if(pInput)
    m_cInputList.Release(strInput);

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Resume(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strPgrm = cRequest.GetArg("program");
  ASSERT(strPgrm != "");

  C_String strInput = cRequest.GetArg("input");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
  }
  else
  {
    m_cBroadcasts.Lock();
    C_Broadcast* pBroadcast = m_cBroadcasts.Find(strPgrm + ":" + strInput);
    if (pBroadcast)
    {
      C_Answer cInputAnswer = pInput->ResumeStreaming(pBroadcast);
      if(cInputAnswer.GetStatus())
      {
        cAnswer.SetStatus(cInputAnswer.GetStatus());
      }
      cAnswer.Add(cInputAnswer);

      // Release repository items (Channel, Program, Input)
      m_cChannelList.Release(pBroadcast->GetChannel()->GetName());
      m_cProgramList.ReleasePgrm(strPgrm);
      m_cInputList.Release(strInput);
    }
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Pgrm was not broadcasted");
    }
    m_cBroadcasts.UnLock();
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Suspend(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strPgrm = cRequest.GetArg("program");
  ASSERT(strPgrm != "");

  C_String strInput = cRequest.GetArg("input");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
  }
  else
  {
    m_cBroadcasts.Lock();
    C_Broadcast* pBroadcast = m_cBroadcasts.Find(strPgrm + ":" + strInput);
    if (pBroadcast)
    {
      C_Answer cInputAnswer = pInput->SuspendStreaming(pBroadcast);
      if(cInputAnswer.GetStatus())
      {
        cAnswer.SetStatus(cInputAnswer.GetStatus());
      }
      cAnswer.Add(cInputAnswer);

      // Release repository items (Channel, Program, Input)
      m_cChannelList.Release(pBroadcast->GetChannel()->GetName());
      m_cProgramList.ReleasePgrm(strPgrm);
      m_cInputList.Release(strInput);
    }
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Pgrm was not broadcasted");
    }
    m_cBroadcasts.UnLock();
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Stop(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strPgrm = cRequest.GetArg("program");
  ASSERT(strPgrm != "");

  C_String strInput = cRequest.GetArg("input");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
  }
  else
  {
    m_cBroadcasts.Lock();
    C_Broadcast* pBroadcast = m_cBroadcasts.Find(strPgrm + ":" + strInput);
    if (pBroadcast)
    {
      C_Answer cInputAnswer = pInput->StopStreaming(pBroadcast);
      if(cInputAnswer.GetStatus())
      {
        cAnswer.SetStatus(cInputAnswer.GetStatus());
      }
      cAnswer.Add(cInputAnswer);

      // Release repository items (Channel, Program, Input)
      m_cChannelList.Release(pBroadcast->GetChannel()->GetName());
      m_cProgramList.ReleasePgrm(strPgrm);
      m_cInputList.Release(strInput);

      LogDbg(m_hLog, "Remove the broadcast");
      int iRc = m_cBroadcasts.Remove(strPgrm + ":" + strInput);
      LogDbg(m_hLog, C_String("Broadcast removed with status ") + iRc);
    }
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Pgrm was not broadcasted");
    }
    m_cBroadcasts.UnLock();
  }

  return cAnswer;
}

