/*****************************************************************************
 * gyachi_lib.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2003 Erica Andrews (PhrozenSmoke ['at'] yahoo.com)
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * common routines for gyachi, and assorted utilities
 *****************************************************************************/


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libintl.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <gtk/gtk.h>

#include "gyachi_lib.h"
#include "gy_config.h"
#include "theme_support.h"


/*        For historical notes, protocol numbers       */
/*  0x09000000 - old, YMSG-9                           */
/*  0x0A000000 - old YMSG10                            */ 
/*  0x0a00c800 - latest  'official' Linux Yahoo client */ 
/*  0x000B0000 - current: YMSG-11, Messenger 5.6       */
/*  0x000C0000 - current: YMSG-12, Messenger 6 Beta    */
/*  0x000D0000 - current: YMSG-13, Messenger 7.5 Beta  */
/*  0x000F0000 - current: YMSG-15, Messenger 8.0 Beta  */
/*  0x00100000 - current: YMSG-16, Messenger 9.0       */

YMSG_PROTOCOL YMSG_PROTOCOLS[] = {
/*{ YMSG_VER_11, "YMSG-11 / Messenger 5.6",      "5, 6, 0, 1358", "???????" }, */
/*{ YMSG_VER_13, "YMSG-13 / Messenger 7.5 Beta", "7,5,0333",      "???????" }, */
  { YMSG_VER_15, "YMSG-15 / Messenger 8",        "8.1.0.421",     "2097087" },
  { YMSG_VER_16, "YMSG-16 / Messenger 9",        "9.0.0.2152",    "4194239" },
  { 0,          0 }
};

char *_(char *some) 
{
/*
#ifdef USE_GTK2
	static gchar *my_locale_string=NULL;

	if (my_locale_string != NULL) {g_free(my_locale_string);}
	my_locale_string=g_locale_to_utf8(gettext(some),-1, NULL, NULL, NULL );
	return my_locale_string;
#endif */
	return gettext(some);
}

gchar *_utf(char *some) { /* converts from the assumed locale to UTF */
	static gchar *mygconvedstr=NULL;

	if (mygconvedstr) {g_free(mygconvedstr); mygconvedstr=NULL;}
	if (g_utf8_validate ( some, -1, NULL))  { /* already in utf */
		mygconvedstr=g_strdup(some);
		return mygconvedstr;
	}
	mygconvedstr=g_convert(some, -1,"UTF-8","ISO8859-1",NULL,NULL,NULL);
	if (! mygconvedstr) {mygconvedstr=g_strdup(some);}
	return mygconvedstr;
}

gchar *_b2loc(char *some) {  /* converts back from UTF to locale string */
	static gchar *my_locale_string=NULL;

	if (my_locale_string) {g_free(my_locale_string); my_locale_string=NULL;}
	if (! g_utf8_validate ( some, -1, NULL))  {
		my_locale_string=g_strdup(some);
		return my_locale_string;
	}
	my_locale_string=g_locale_from_utf8(some, -1, NULL, NULL,NULL);
	if (! my_locale_string) {my_locale_string=g_strdup(some);}
	return my_locale_string;
}

void lower_str( char *str ) {
	char *dst = str;
	char *src = str;

	while (*src) {
		*(dst++) = tolower(*(src++));
 	}
}

YMSG_PROTOCOL *yprotocol_from_proto_id(int proto)
{
	YMSG_PROTOCOL *y_proto;

	for (y_proto = YMSG_PROTOCOLS; y_proto->protocol; y_proto++) {
		if (y_proto->protocol == proto) return(y_proto);
	}

	return 0;
}

int yprotocol_name_to_protocol(const char *proto)
{
	YMSG_PROTOCOL *y_proto;

	for (y_proto = YMSG_PROTOCOLS; y_proto->protocol; y_proto++) {
		if (strcmp(y_proto->proto_name, proto) == 0) return(y_proto->protocol);
	}

	return 0;
}

const char *yprotocol_to_name(int proto)
{
	YMSG_PROTOCOL *y_proto;

	y_proto = yprotocol_from_proto_id(proto);
	if (y_proto) return(y_proto->proto_name);

	return 0;
}


/* common code very commonly used to set a tooltip */
void gyachi_set_tooltip(GtkWidget *somewid, char *somechar) {
#ifdef HAVE_GTK_WIDGET_TOOLTIP
	gtk_widget_set_tooltip_text(somewid, somechar);
#else
	GtkTooltips *TIP_GROUP;

	if (!somewid) {return;}
	TIP_GROUP = GTK_TOOLTIPS(g_object_get_data(G_OBJECT(somewid), "tooltip_group"));
	if ( TIP_GROUP == NULL ) {
		TIP_GROUP = gtk_tooltips_new();
		g_object_set_data(G_OBJECT(somewid), "tooltip_group", TIP_GROUP);
	}

	if (!somechar) {
		gtk_tooltips_disable(TIP_GROUP);
		return;
	}

	gtk_tooltips_set_tip(TIP_GROUP, somewid, somechar, somechar);
	gtk_tooltips_enable(TIP_GROUP);
#endif
}


GtkWidget *gyachi_combobox_new(int editable)
{
	GtkWidget *gy_combo_box;

	if (editable) {
		gy_combo_box = gtk_combo_box_entry_new_text();
	}
	else {
		gy_combo_box = gtk_combo_box_new_text();
	}
	gtk_widget_show (gy_combo_box);
	return gy_combo_box;
}

void gyachi_combobox_load(GtkWidget *gy_combo_box, char *item_array[])
{
	char  **item_pointer;

	for (item_pointer = item_array; *item_pointer; item_pointer++) {
		gtk_combo_box_append_text(GTK_COMBO_BOX(gy_combo_box), *item_pointer);
	}
}

void gyachi_combobox_load_list(GtkWidget *gy_combo_box, GList *item_list)
{
	GList *this_item;
	char  *item;

	for (this_item=item_list; this_item; this_item=g_list_next(this_item)) {
		item=(char *)this_item->data;
		gtk_combo_box_append_text(GTK_COMBO_BOX(gy_combo_box), item);
	}
}

void gyachi_combobox_prepend_item(GtkWidget *gy_combo_box, const char *item) {
	gtk_combo_box_prepend_text(GTK_COMBO_BOX(gy_combo_box), item);
}

void gyachi_combobox_append_item(GtkWidget *gy_combo_box, const char *item) {
	gtk_combo_box_append_text(GTK_COMBO_BOX(gy_combo_box), item);
}

void gyachi_combobox_select_itemno(GtkWidget *gy_combo_box, int selected_item_no)
{
	gtk_combo_box_set_active(GTK_COMBO_BOX(gy_combo_box), selected_item_no);
}

/* returns 1 (TRUE) if item is found/selected, 0 (FALSE) otherwise */
int gyachi_combobox_select_item(GtkWidget *gy_combo_box, char *item)
{
	GtkTreeModel *model;
	GtkTreeIter   combobox_iter;
	gboolean      valid;
	gchar        *text = NULL;
	int           selected_item_no;
	int           found = FALSE;

	g_object_get(GTK_COMBO_BOX(gy_combo_box), "model", &model, NULL);
	valid = gtk_tree_model_get_iter_first(model, &combobox_iter);
	selected_item_no = 0;
	while (valid) {
		gtk_tree_model_get(model, &combobox_iter, 0, &text, -1);
		if (!strcmp(text, item)) {
			found=TRUE;
			g_free(text);
			break;
		}
		g_free(text);
		selected_item_no++;
		valid = gtk_tree_model_iter_next(model, &combobox_iter);
	}
	if (found) {
		gyachi_combobox_select_itemno(gy_combo_box, selected_item_no);
	}

	return(found);
}

const char *gyachi_combobox_get_selected_item(GtkWidget *gy_combo_box)
{
	static char *text = NULL;

	if (text) {
		free(text);
		text = NULL;
	}

	text = gtk_combo_box_get_active_text(GTK_COMBO_BOX(gy_combo_box));
	return(text);
}

#if 0
// This is the original (pre 1.1.62) lookup widget
// I replaced it with the util.c lookup_widget in 1.1.62 to
// consolidate the 3 versions of lookup_widget (util.c, gyachi_lib.c, and gyvoice/support.c
// Now all everyone uses the util.c version.
//
// The basic difference is that util.c looks in the present object for
// the string, whereas all others immediately go to, and start at, the
// parent. As I recall, util.c originally went to parent first, and this
// caused problems when the actual object was in the immediate object,
// so the order of checking was changed to look in the object first,
// *before* going to the parent.
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name)
{
	GtkWidget *parent, *found_widget;

	for (;;) {
		if (GTK_IS_MENU (widget)) {
			parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
		}
		else {
			parent = widget->parent;
		}
		if (!parent) {
			parent = (GtkWidget*) g_object_get_data (G_OBJECT (widget), "GladeParentKey");
		}
		if (parent == NULL) {
			break;
		}
		widget = parent;
	}

	found_widget = (GtkWidget*) g_object_get_data (G_OBJECT (widget),
                                                 widget_name);
	if (!found_widget) {
		g_warning ("Widget not found: %s", widget_name);
	}
	return found_widget;
}
#else

/* This was moved in from the no-longer-used client/support.c module */

GtkWidget *lookup_widget (GtkWidget  *widget, const gchar *widget_name)
{
  GtkWidget *parent, *found_widget;

  for (;;)
    {
      found_widget = (GtkWidget*) g_object_get_data (G_OBJECT (widget),
						       widget_name);
      if (found_widget) {
	return(found_widget);
      }
      
      if (GTK_IS_MENU (widget))
        parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
      else
        parent = widget->parent;
      if (parent == NULL)
        break;
      widget = parent;
    }

  g_warning ("Widget not found: %s", widget_name);
  return(NULL);
}
#endif

YMSG_ROOM *find_room(char *room_name) {
	GList     *this_item;
	YMSG_ROOM *room;

	for (this_item=room_list; this_item; this_item=g_list_next(this_item)) {
		room=(YMSG_ROOM*)this_item->data;
		if (!strcmp(room_name, room->room_name)) {
			return(room);
		}
	}
	return NULL;
}

void add_room(char *room_name, char *room_id, int fav) {
	YMSG_ROOM *room;

	room = find_room(room_name);
	if (room) {
		if (room_id) {
			free(room->room_id);
			room->room_id = strdup(room_id);
		}
		if (fav >= 0) {
			room->fav = fav;
		}

		return;
	}

	room = malloc(sizeof(YMSG_ROOM));
	room->room_name = strdup(room_name);
	room->room_id   = strdup(room_id?room_id:"0");
	room->fav       = (fav>=0)?fav:0;
	room_list = g_list_append(room_list, room);
}

/* NOTE: This function dynamically malloc's the memory
 * for the filename,so be sure to free the returned result.
 * NOTE: The last array entry must be a NULL pointer...
 */
char *build_string(char **strngs) {
	int len;
	int i;
	char *rv;

	for (len=1, i=0; strngs[i]; i++) {
		len += strlen(strngs[i]);
	}

	rv = malloc(len);

	for (len=0, i=0; strngs[i]; i++) {
		strcpy(&rv[len], strngs[i]);
		len += strlen(strngs[i]);
	}
	rv[len]=0;

	return(rv);
}



char *gyachi_filename(char *(pieces[])) {
	return(build_string(pieces));
}

int max_linelen(FILE *fp) {
	int max;
	int i;
	int c;

	rewind(fp);
	for (max=0, i=1; (c=fgetc(fp)) != EOF; i++) {
		if (c == '\n') {
			if (i > max) max = i;
			i=0;
		}
	}
	if (i > max) max = i;

	rewind(fp);
	clearerr(fp);
	if (max == 1) max++;
	return(max);
}

void on_close_ok_message(GtkWidget *widget, gpointer  user_data)
{
	GtkWidget *messagewin = (GtkWidget *)user_data;
	typedef void (*CALLBACK_FUNCTION)();
	CALLBACK_FUNCTION callback;

	callback = g_object_get_data(G_OBJECT(messagewin), "callback");
	gtk_widget_destroy(messagewin);
	if (callback) {
		callback();
	}
}

void on_close_ok_messagew(GtkWidget *widget, GdkEvent *event, gpointer  user_data)
{
	on_close_ok_message(widget, user_data);
}



GtkWidget *get_pixmapped_button(char *button_label, const gchar *stock_id) {
	GtkWidget *buttbox;
	GtkWidget *dbutton;
	GtkWidget *dlab;

	dlab=gtk_label_new(button_label);
  	gtk_label_set_justify (GTK_LABEL (dlab), GTK_JUSTIFY_LEFT);
  	gtk_misc_set_alignment (GTK_MISC (dlab), 0, 0.5);

	dbutton=gtk_button_new();
	buttbox=gtk_hbox_new(FALSE, 2);
  	gtk_container_set_border_width (GTK_CONTAINER (buttbox), 1);
	gtk_box_pack_start (GTK_BOX (buttbox), GTK_WIDGET(gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_BUTTON) ), FALSE, FALSE, 1);
	gtk_box_pack_start (GTK_BOX (buttbox), GTK_WIDGET(dlab), TRUE, TRUE,2);
	gtk_container_add (GTK_CONTAINER (dbutton), buttbox); 
	gtk_widget_show_all(dbutton);
	return dbutton;
}

void show_ok_message(char *mymsg, GtkWidget *parent, char *ui_name, int error, void *callback)  {
	GtkWidget *dwindow;
	GtkWidget *dvbox;
	GtkWidget *dlbox;
	GtkWidget *dlabel;	
	GtkWidget *dbutton;
	GdkPixbuf *pixbuf;
	char  *(pieces[5]);
	char *cammsg;

	if (ui_name) {
		pieces[0]="GyachI  (";
		pieces[1]=ui_name;
		pieces[2]="):\n\n";
		pieces[3]=mymsg;
		pieces[4]=NULL;
	}
	else {
		pieces[0]=mymsg;
		pieces[1]=NULL;
	}
	cammsg=build_string(pieces);

	dwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	g_object_set_data(G_OBJECT (dwindow), "mywindow", dwindow);
	gtk_window_set_title (GTK_WINDOW (dwindow), _("GYachI: Message"));
	gtk_window_set_resizable(GTK_WINDOW (dwindow), TRUE);
	if (parent) {
		gtk_window_set_transient_for(GTK_WINDOW(dwindow), GTK_WINDOW(parent));
		gtk_window_set_destroy_with_parent(GTK_WINDOW(dwindow), TRUE);
		gtk_window_set_position(GTK_WINDOW(dwindow), GTK_WIN_POS_CENTER_ON_PARENT);
		gtk_window_present(GTK_WINDOW(parent));
	}
	else {
		gtk_window_set_position(GTK_WINDOW(dwindow), GTK_WIN_POS_CENTER);
	}

	dvbox = gtk_vbox_new (FALSE, 2);
	gtk_container_add (GTK_CONTAINER (dwindow), dvbox); 
	gtk_container_set_border_width (GTK_CONTAINER (dvbox), 6);  

	dlabel=gtk_label_new("");
	gtk_label_set_text(GTK_LABEL(dlabel),cammsg); /* for gettext support, gchar */
	gtk_label_set_line_wrap(GTK_LABEL(dlabel),1);
	free(cammsg);

	dbutton=get_pixmapped_button(_("OK"), GTK_STOCK_YES);	
	g_object_set_data(G_OBJECT (dbutton), "mywindow", dwindow);

	dlbox = gtk_hbox_new (FALSE, 2);

	if (error) {
  		gtk_box_pack_start (GTK_BOX (dlbox), GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_DIALOG_ERROR, GTK_ICON_SIZE_DIALOG)), FALSE, FALSE , 2);
	}
	else {
  		gtk_box_pack_start (GTK_BOX (dlbox), GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG)), FALSE, FALSE , 2);
	}

  	gtk_box_pack_start (GTK_BOX (dlbox), dlabel, TRUE, TRUE, 4);
  	gtk_box_pack_start (GTK_BOX (dvbox), dlbox, TRUE, TRUE, 4);
	gtk_box_pack_start (GTK_BOX (dvbox), dbutton, FALSE, FALSE, 8);

	g_object_set_data(G_OBJECT(dwindow), "callback", callback);
	g_signal_connect(dbutton, "clicked",      (GCallback)on_close_ok_message,  dwindow);
	g_signal_connect(dwindow, "delete_event", (GCallback)on_close_ok_messagew, dwindow);

	gtk_window_set_modal(GTK_WINDOW(dwindow), 1);
	gtk_widget_show_all(dwindow);

	pixbuf = get_pixbuf_from_stock_id(dwindow, GYACHI_ICON, GTK_ICON_SIZE_LARGE_TOOLBAR);
	if (pixbuf) {
		gtk_window_set_icon( GTK_WINDOW(dwindow), pixbuf );
		g_object_unref(pixbuf);
	}
}

