/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.ssl;

import com.google.common.collect.ImmutableMap;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.OpenSsl;
import io.netty.util.internal.PlatformDependent;
import java.io.IOException;
import java.nio.file.Path;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.crypto.Cipher;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.common.Booleans;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.settings.Settings;
import org.opensearch.env.Environment;
import org.opensearch.security.ssl.OpenSearchSecuritySSLPlugin;
import org.opensearch.security.ssl.SslConfiguration;
import org.opensearch.security.ssl.SslContextHandler;
import org.opensearch.security.ssl.config.CertType;
import org.opensearch.security.ssl.config.KeyStoreConfiguration;
import org.opensearch.security.ssl.config.SslCertificatesLoader;
import org.opensearch.security.ssl.config.SslParameters;
import org.opensearch.security.ssl.config.TrustStoreConfiguration;
import org.opensearch.watcher.FileChangesListener;
import org.opensearch.watcher.FileWatcher;
import org.opensearch.watcher.ResourceWatcher;
import org.opensearch.watcher.ResourceWatcherService;

public class SslSettingsManager {
    private static final Logger LOGGER = LogManager.getLogger(SslSettingsManager.class);
    private final Map<CertType, SslContextHandler> sslSettingsContexts;

    public SslSettingsManager(Environment environment) {
        this.sslSettingsContexts = this.buildSslContexts(environment);
    }

    public Optional<SslConfiguration> sslConfiguration(CertType certType) {
        return Optional.ofNullable(this.sslSettingsContexts.get((Object)certType)).map(SslContextHandler::sslConfiguration);
    }

    public Optional<SslContextHandler> sslContextHandler(CertType sslConfigPrefix) {
        return Optional.ofNullable(this.sslSettingsContexts.get((Object)sslConfigPrefix));
    }

    private Map<CertType, SslContextHandler> buildSslContexts(Environment environment) {
        ImmutableMap.Builder contexts = new ImmutableMap.Builder();
        Map<CertType, SslConfiguration> configurations = this.loadConfigurations(environment);
        Optional.ofNullable(configurations.get((Object)CertType.HTTP)).ifPresentOrElse(sslConfiguration -> contexts.put((Object)CertType.HTTP, (Object)new SslContextHandler((SslConfiguration)sslConfiguration)), () -> LOGGER.warn("SSL Configuration for HTTP Layer hasn't been set"));
        Optional.ofNullable(configurations.get((Object)CertType.TRANSPORT)).ifPresentOrElse(sslConfiguration -> {
            contexts.put((Object)CertType.TRANSPORT, (Object)new SslContextHandler((SslConfiguration)sslConfiguration));
            SslConfiguration transportClientConfiguration = Optional.ofNullable((SslConfiguration)configurations.get((Object)CertType.TRANSPORT_CLIENT)).orElse((SslConfiguration)sslConfiguration);
            contexts.put((Object)CertType.TRANSPORT_CLIENT, (Object)new SslContextHandler(transportClientConfiguration, true));
        }, () -> LOGGER.warn("SSL Configuration for Transport Layer hasn't been set"));
        return contexts.build();
    }

    public synchronized void reloadSslContext(CertType certType) {
        this.sslContextHandler(certType).ifPresentOrElse(sscContextHandler -> {
            try {
                if (sscContextHandler.reloadSslContext()) {
                    LOGGER.info("{} SSL context reloaded", (Object)certType.name());
                }
            }
            catch (CertificateException e) {
                throw new OpenSearchException((Throwable)e);
            }
        }, () -> LOGGER.error("Missing SSL Context for {}", (Object)certType.name()));
    }

    private Map<CertType, SslConfiguration> loadConfigurations(Environment environment) {
        Settings settings = environment.settings();
        Settings httpSettings = settings.getByPrefix(CertType.HTTP.sslConfigPrefix());
        Settings transportSettings = settings.getByPrefix(CertType.TRANSPORT.sslConfigPrefix());
        if (httpSettings.isEmpty() && transportSettings.isEmpty()) {
            throw new OpenSearchException("No SSL configuration found", new Object[0]);
        }
        this.jceWarnings();
        this.openSslWarnings(settings);
        Boolean httpEnabled = httpSettings.getAsBoolean("enabled", Boolean.valueOf(false));
        Boolean transportEnabled = transportSettings.getAsBoolean("enabled", Boolean.valueOf(true));
        ImmutableMap.Builder configurationBuilder = ImmutableMap.builder();
        if (httpEnabled.booleanValue() && !this.clientNode(settings)) {
            this.validateHttpSettings(httpSettings);
            SslParameters httpSslParameters = SslParameters.loader(httpSettings).load(true);
            Tuple<TrustStoreConfiguration, KeyStoreConfiguration> httpTrustAndKeyStore = new SslCertificatesLoader(CertType.HTTP.sslConfigPrefix()).loadConfiguration(environment);
            configurationBuilder.put((Object)CertType.HTTP, (Object)new SslConfiguration(httpSslParameters, (TrustStoreConfiguration)httpTrustAndKeyStore.v1(), (KeyStoreConfiguration)httpTrustAndKeyStore.v2()));
            LOGGER.info("TLS HTTP Provider                    : {}", (Object)httpSslParameters.provider());
            LOGGER.info("Enabled TLS protocols for HTTP layer : {}", httpSslParameters.allowedProtocols());
        }
        SslParameters transportSslParameters = SslParameters.loader(transportSettings).load(false);
        if (transportEnabled.booleanValue()) {
            if (this.hasExtendedKeyUsageEnabled(transportSettings)) {
                this.validateTransportSettings(transportSettings);
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportServerTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslConfigPrefix(), "server.").loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportServerTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportServerTrustAndKeyStore.v2()));
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportClientTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslConfigPrefix(), "client.").loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT_CLIENT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportClientTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportClientTrustAndKeyStore.v2()));
            } else {
                this.validateTransportSettings(transportSettings);
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslConfigPrefix()).loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportTrustAndKeyStore.v2()));
            }
            LOGGER.info("TLS Transport Client Provider             : {}", (Object)transportSslParameters.provider());
            LOGGER.info("TLS Transport Server Provider             : {}", (Object)transportSslParameters.provider());
            LOGGER.info("Enabled TLS protocols for Transport layer : {}", transportSslParameters.allowedProtocols());
        }
        return configurationBuilder.build();
    }

    public void addSslConfigurationsChangeListener(ResourceWatcherService resourceWatcherService) {
        for (Path directoryToMonitor : this.directoriesToMonitor()) {
            FileWatcher fileWatcher = new FileWatcher(directoryToMonitor);
            fileWatcher.addListener((Object)new FileChangesListener(){

                public void onFileCreated(Path file) {
                    this.onFileChanged(file);
                }

                public void onFileDeleted(Path file) {
                    this.onFileChanged(file);
                }

                public void onFileChanged(Path file) {
                    for (Map.Entry<CertType, SslContextHandler> e : SslSettingsManager.this.sslSettingsContexts.entrySet()) {
                        CertType certType = e.getKey();
                        SslConfiguration sslConfiguration = e.getValue().sslConfiguration();
                        if (!sslConfiguration.dependentFiles().contains(file)) continue;
                        SslSettingsManager.this.reloadSslContext(certType);
                    }
                }
            });
            try {
                resourceWatcherService.add((ResourceWatcher)fileWatcher, ResourceWatcherService.Frequency.HIGH);
                LOGGER.info("Added SSL configuration change listener for: {}", (Object)directoryToMonitor);
            }
            catch (IOException e) {
                throw new OpenSearchException("Couldn't add SSL configurations change listener", (Throwable)e, new Object[0]);
            }
        }
    }

    private Set<Path> directoriesToMonitor() {
        return this.sslSettingsContexts.values().stream().map(SslContextHandler::sslConfiguration).flatMap(c -> c.dependentFiles().stream()).map(Path::getParent).collect(Collectors.toSet());
    }

    private boolean clientNode(Settings settings) {
        return !"node".equals(settings.get("client.type"));
    }

    private void validateHttpSettings(Settings httpSettings) {
        if (httpSettings == null) {
            return;
        }
        if (!httpSettings.getAsBoolean("enabled", Boolean.valueOf(false)).booleanValue()) {
            return;
        }
        ClientAuth clientAuth = ClientAuth.valueOf((String)httpSettings.get("clientauth_mode", ClientAuth.OPTIONAL.name()).toUpperCase(Locale.ROOT));
        if (this.hasPemStoreSettings(httpSettings)) {
            if (!httpSettings.hasValue("pemcert_filepath") || !httpSettings.hasValue("pemkey_filepath")) {
                throw new OpenSearchException("Wrong HTTP SSL configuration. " + String.join((CharSequence)", ", "plugins.security.ssl.http.pemcert_filepath", "plugins.security.ssl.http.pemkey_filepath") + " must be set", new Object[0]);
            }
            if (clientAuth == ClientAuth.REQUIRE && !httpSettings.hasValue("pemtrustedcas_filepath")) {
                throw new OpenSearchException("Wrong HTTP SSL configuration. plugins.security.ssl.http.pemtrustedcas_filepath must be set if client auth is required", new Object[0]);
            }
        } else if (this.hasKeyOrTrustStoreSettings(httpSettings)) {
            if (!httpSettings.hasValue("keystore_filepath")) {
                throw new OpenSearchException("Wrong HTTP SSL configuration. plugins.security.ssl.http.keystore_filepath must be set", new Object[0]);
            }
            if (clientAuth == ClientAuth.REQUIRE && !httpSettings.hasValue("truststore_filepath")) {
                throw new OpenSearchException("Wrong HTTP SSL configuration. plugins.security.ssl.http.truststore_filepath must be set if client auth is required", new Object[0]);
            }
        } else {
            throw new OpenSearchException("Wrong HTTP SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure HTTP layer", new Object[0]);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void validateTransportSettings(Settings transportSettings) {
        if (!this.hasExtendedKeyUsageEnabled(transportSettings)) {
            if (this.hasPemStoreSettings(transportSettings)) {
                if (transportSettings.hasValue("pemcert_filepath") && transportSettings.hasValue("pemkey_filepath") && transportSettings.hasValue("pemtrustedcas_filepath")) return;
                throw new OpenSearchException("Wrong Transport SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.pemcert_filepath", "plugins.security.ssl.transport.pemkey_filepath", "plugins.security.ssl.transport.pemtrustedcas_filepath") + " must be set", new Object[0]);
            }
            if (!this.hasKeyOrTrustStoreSettings(transportSettings)) throw new OpenSearchException("Wrong Transport SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure Transport layer properly", new Object[0]);
            this.verifyKeyAndTrustStoreSettings(transportSettings);
            return;
        } else {
            Settings serverTransportSettings = transportSettings.getByPrefix("server.");
            Settings clientTransportSettings = transportSettings.getByPrefix("client.");
            if (this.hasKeyOrTrustStoreSettings(transportSettings)) {
                this.verifyKeyAndTrustStoreSettings(transportSettings);
                if (serverTransportSettings.hasValue("keystore_alias") && serverTransportSettings.hasValue("truststore_alias") && clientTransportSettings.hasValue("keystore_alias") && clientTransportSettings.hasValue("truststore_alias")) return;
                throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.server.keystore_alias", "plugins.security.ssl.transport.server.truststore_alias", "plugins.security.ssl.transport.client.keystore_alias", "plugins.security.ssl.transport.client.truststore_alias") + " must be set if plugins.security.ssl.transport.extended_key_usage_enabled is set", new Object[0]);
            }
            if (this.hasKeyOrTrustStoreSettings(transportSettings)) throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure HTTP layer", new Object[0]);
            if (serverTransportSettings.hasValue("pemcert_filepath") && serverTransportSettings.hasValue("pemkey_filepath") && serverTransportSettings.hasValue("pemtrustedcas_filepath") && clientTransportSettings.hasValue("pemcert_filepath") && clientTransportSettings.hasValue("pemkey_filepath") && clientTransportSettings.hasValue("pemtrustedcas_filepath")) return;
            throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.server.pemcert_filepath", "plugins.security.ssl.transport.server.pemkey_filepath", "plugins.security.ssl.transport.server.pemtrustedcas_filepath", "plugins.security.ssl.transport.client.pemcert_filepath", "plugins.security.ssl.transport.client.pemkey_filepath", "plugins.security.ssl.transport.client.pemtrustedcas_filepath") + " must be set if plugins.security.ssl.transport.extended_key_usage_enabled is set", new Object[0]);
        }
    }

    private void verifyKeyAndTrustStoreSettings(Settings settings) {
        if (!settings.hasValue("keystore_filepath") || !settings.hasValue("truststore_filepath")) {
            throw new OpenSearchException("Wrong Transport/Tran SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure Transport layer properly", new Object[0]);
        }
    }

    private boolean hasExtendedKeyUsageEnabled(Settings settings) {
        return settings.getAsBoolean("extended_key_usage_enabled", Boolean.valueOf(false));
    }

    private boolean hasKeyOrTrustStoreSettings(Settings settings) {
        return settings.hasValue("keystore_filepath") || settings.hasValue("truststore_filepath");
    }

    private boolean hasPemStoreSettings(Settings settings) {
        return settings.hasValue("pemkey_filepath") || settings.hasValue("pemcert_filepath") || settings.hasValue("pemtrustedcas_filepath");
    }

    void jceWarnings() {
        try {
            int aesMaxKeyLength = Cipher.getMaxAllowedKeyLength("AES");
            if (aesMaxKeyLength < 256) {
                LOGGER.info("AES-256 not supported, max key length for AES is {} bit. (This is not an issue, it just limits possible encryption strength. To enable AES 256, install 'Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy Files')", (Object)aesMaxKeyLength);
            }
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.error("AES encryption not supported (SG 1). ", (Throwable)e);
        }
    }

    void openSslWarnings(Settings settings) {
        if (!OpenSearchSecuritySSLPlugin.OPENSSL_SUPPORTED && OpenSsl.isAvailable() && (settings.getAsBoolean("plugins.security.ssl.http.enable_openssl_if_available", Boolean.valueOf(true)).booleanValue() || settings.getAsBoolean("plugins.security.ssl.transport.enable_openssl_if_available", Boolean.valueOf(true)).booleanValue())) {
            if (PlatformDependent.javaVersion() < 12) {
                LOGGER.warn("Support for OpenSSL with Java 11 or prior versions require using Netty allocator. Set 'opensearch.unsafe.use_netty_default_allocator' system property to true");
            } else {
                LOGGER.warn("Support for OpenSSL with Java 12+ has been removed from OpenSearch Security. Using JDK SSL instead.");
            }
        }
        if (OpenSearchSecuritySSLPlugin.OPENSSL_SUPPORTED && OpenSsl.isAvailable()) {
            LOGGER.info("OpenSSL {} ({}) available", (Object)OpenSsl.versionString(), (Object)OpenSsl.version());
            if ((long)OpenSsl.version() < 0x10002000L) {
                LOGGER.warn("Outdated OpenSSL version detected. You should update to 1.0.2k or later. Currently installed: {}", (Object)OpenSsl.versionString());
            }
            if (!OpenSsl.supportsHostnameValidation()) {
                LOGGER.warn("Your OpenSSL version {} does not support hostname verification. You should update to 1.0.2k or later.", (Object)OpenSsl.versionString());
            }
            LOGGER.debug("OpenSSL available ciphers {}", (Object)OpenSsl.availableOpenSslCipherSuites());
        } else {
            boolean openSslIsEnabled = false;
            if (settings.hasValue("plugins.security.ssl.http.enable_openssl_if_available")) {
                openSslIsEnabled |= Booleans.parseBoolean((String)settings.get("plugins.security.ssl.http.enable_openssl_if_available"));
            }
            if (settings.hasValue("plugins.security.ssl.transport.enable_openssl_if_available")) {
                openSslIsEnabled |= Booleans.parseBoolean((String)settings.get("plugins.security.ssl.transport.enable_openssl_if_available"));
            }
            if (openSslIsEnabled) {
                LOGGER.warn("OpenSSL not available (this is not an error, we simply fallback to built-in JDK SSL) because of ", OpenSsl.unavailabilityCause());
            }
        }
    }
}

