load ../common

# shellcheck disable=SC2034
tag='ch-image --force'

setup () {
    [[ $CH_BUILDER = ch-image ]] || skip 'ch-image only'
}

@test "${tag}: no matching distro" {
    scope standard

    # without --force
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM alpine:3.9
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'--force not available (no suitable config found)'* ]]

    # with --force
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM alpine:3.9
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'--force not available (no suitable config found)'* ]]
}

@test "${tag}: --no-force-detect" {
    scope standard

    run ch-image -v build --no-force-detect -t fakeroot-temp -f - . <<'EOF'
FROM alpine:3.9
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'not detecting --force config, per --no-force-detect'* ]]

}

@test "${tag}: misc errors" {
    scope standard

    run ch-image build --force --no-force-detect .
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = 'error'*'are incompatible'* ]]
}

@test "${tag}: multiple RUN" {
    scope standard

    # 1. List form of RUN.
    # 2. apt-get not at beginning.
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN true
RUN true && apt-get update
RUN ["apt-get", "install", "-y", "hello"]
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $(echo "$output" | grep -Fc 'init step 1: checking: $') -eq 1 ]]
    [[ $(echo "$output" | grep -Fc 'init step 1: $') -eq 1 ]]
    [[ $(echo "$output" | grep -Fc 'RUN: new command:') -eq 2 ]]
    [[ $output = *'init: already initialized'* ]]
    [[ $output = *'--force: init OK & modified 2 RUN instructions'* ]]
    [[ $output = *'grown in 4 instructions: fakeroot-temp'* ]]
}

@test "${tag}: CentOS 7: unneeded, no --force, build succeeds" {
    scope standard
    # no commands that may need it, without --force, build succeeds
    # also: correct config, last config tested is the one selected
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force: rhel7'* ]]
    [[ $output = *$'testing config: rhel7\navailable --force'* ]]
}

@test "${tag}: CentOS 7: unneeded, no --force, build fails" {
    scope full
    # no commands that may need it, without --force, build fails
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN false
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *"build failed: current version of --force wouldn't help"* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: CentOS 7: unneeded, with --force" {
    scope full
    # no commands that may need it, with --force, warning
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'warning: --force specified, but nothing to do'* ]]
}

@test "${tag}: CentOS 7: maybe needed but actually not, no --force" {
    scope full
    # commands that may need it, but turns out they don’t, without --force
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN yum install -y ed
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
}

@test "${tag}: CentOS 7: maybe needed but actually not, with --force" {
    scope full
    # commands that may need it, but turns out they don’t, with --force
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN yum install -y ed
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
}

@test "${tag}: CentOS 7: needed but no --force" {
    scope full
    # commands that may need it, they do, fail & suggest
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN yum install -y openssh
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
    [[ $output = *'build failed: --force may fix it'* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: CentOS 7: needed, with --force" {
    scope standard
    # commands that may need it, they do, --force, success
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:7
RUN yum install -y openssh
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
}

@test "${tag}: CentOS 7: EPEL already enabled" {
    scope standard

    # 7: install EPEL (no fakeroot)
    run ch-image -v build -t centos7-epel1 -f - . <<'EOF'
FROM centos:7
RUN yum install -y epel-release
RUN yum repolist | egrep '^epel/'
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force'* ]]
    echo "$output" | grep -E 'Installing.+: epel-release'

    # 7: install openssh (with fakeroot)
    run ch-image -v build --force -t centos7-epel2 -f - . <<'EOF'
FROM centos7-epel1
RUN yum install -y openssh
RUN yum repolist | egrep '^epel/'
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 2 RUN instructions'* ]]
    ! ( echo "$output" | grep -E '(Updating|Installing).+: epel-release' )
}

@test "${tag}: CentOS 8: unneeded, no --force, build succeeds" {
    scope standard
    # no commands that may need it, without --force, build succeeds
    # also: correct config
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force: rhel8'* ]]
}

@test "${tag}: CentOS 8: unneeded, no --force, build fails" {
    scope standard
    # no commands that may need it, without --force, build fails
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN false
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *"build failed: current version of --force wouldn't help"* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: CentOS 8: unneeded, with --force" {
    scope standard
    # no commands that may need it, with --force, warning
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'warning: --force specified, but nothing to do'* ]]
}

@test "${tag}: CentOS 8: maybe needed but actually not, no --force" {
    scope standard
    # commands that may need it, but turns out they don’t, without --force
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN dnf install -y ed
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
}

@test "${tag}: CentOS 8: maybe needed but actually not, with --force" {
    scope standard
    # commands that may need it, but turns out they don’t, with --force
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN dnf install -y ed
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
}

@test "${tag}: CentOS 8: needed but no --force" {
    scope standard
    # commands that may need it, they do, fail & suggest
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN dnf install -y --setopt=install_weak_deps=false openssh
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
    [[ $output = *'build failed: --force may fix it'* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: CentOS 8: needed, with --force" {
    scope standard
    # commands that may need it, they do, --force, success
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM centos:8
RUN dnf install -y --setopt=install_weak_deps=false openssh
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
    # validate EPEL is installed but not enabled
    ls -lh "$CH_IMAGE_STORAGE"/img/fakeroot-temp/etc/yum.repos.d/epel*.repo
    ! grep -Eq 'enabled=1' "$CH_IMAGE_STORAGE"/img/fakeroot-temp/etc/yum.repos.d/epel*.repo
}

@test "${tag}: CentOS 8: EPEL already installed" {
    scope standard

    # install EPEL, no --force
    run ch-image -v build -t epel1 -f - . <<'EOF'
FROM centos:8
RUN dnf install -y epel-release
RUN dnf repolist | egrep '^epel'
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force'* ]]
    echo "$output" | grep -E 'Installing.+: epel-release'

    # new image based on that
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM epel1
RUN dnf install -y openssh
RUN dnf repolist | egrep '^epel'
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 2 RUN instructions'* ]]
    ! ( echo "$output" | grep -E '(Updating|Installing).+: epel-release' )
    # validate EPEL is installed *and* enabled
    ls -lh "$CH_IMAGE_STORAGE"/img/fakeroot-temp/etc/yum.repos.d/epel*.repo
    grep -Eq 'enabled=1' "$CH_IMAGE_STORAGE"/img/fakeroot-temp/etc/yum.repos.d/epel*.repo
}

@test "${tag}: Debian Stretch: unneeded, no --force, build succeeds" {
    scope standard
    # no commands that may need it, without --force, build succeeds
    # also: correct config
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:stretch
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force: debSB'* ]]
}

@test "${tag}: Debian Stretch: unneeded, no --force, build fails" {
    scope full
    # no commands that may need it, without --force, build fails
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:stretch
RUN false
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *"build failed: current version of --force wouldn't help"* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: Debian Stretch: unneeded, with --force" {
    scope full
    # no commands that may need it, with --force, warning
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM debian:stretch
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'warning: --force specified, but nothing to do'* ]]
}

# FIXME: Not sure how to do this on Debian; any use of apt-get to install
# needs "apt-get update" first, which requires --force.
#@test "${tag}: Debian Stretch: maybe needed but actually not, no --force" {
#}

# FIXME: Not sure how to do this on Debian; any use of apt-get to install
# needs "apt-get update" first, which requires --force.
#@test "${tag}: Debian Stretch: maybe needed but actually not, with --force" {
#}

@test "${tag}: Debian Stretch: needed but no --force" {
    scope full
    # commands that may need it, they do, fail & suggest
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:stretch
RUN apt-get update && apt-get install -y openssh-client
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
    [[ $output = *'build failed: --force may fix it'* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: Debian Stretch: needed, with --force" {
    scope full
    # commands that may need it, they do, --force, success
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM debian:stretch
RUN apt-get update && apt-get install -y openssh-client
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
}

@test "${tag}: Debian Buster: unneeded, no --force, build succeeds" {
    scope standard
    # no commands that may need it, without --force, build succeeds
    # also: correct config
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'available --force: debSB'* ]]
}

@test "${tag}: Debian Buster: unneeded, no --force, build fails" {
    scope full
    # no commands that may need it, without --force, build fails
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN false
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *"build failed: current version of --force wouldn't help"* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: Debian Buster: unneeded, with --force" {
    scope full
    # no commands that may need it, with --force, warning
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN true
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'warning: --force specified, but nothing to do'* ]]
}

# FIXME: Not sure how to do this on Debian; any use of apt-get to install
# needs "apt-get update" first, which requires --force.
#@test "${tag}: Debian Stretch: maybe needed but actually not, no --force" {
#}

# FIXME: Not sure how to do this on Debian; any use of apt-get to install
# needs "apt-get update" first, which requires --force.
#@test "${tag}: Debian Stretch: maybe needed but actually not, with --force" {
#}

@test "${tag}: Debian Buster: needed but no --force" {
    scope full
    # commands that may need it, they do, fail & suggest
    run ch-image -v build -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN apt-get update && apt-get install -y openssh-client
EOF
    echo "$output"
    [[ $status -eq 1 ]]
    [[ $output = *'available --force'* ]]
    [[ $output = *'RUN: available here with --force'* ]]
    [[ $output = *'build failed: --force may fix it'* ]]
    [[ $output = *'build failed: RUN command exited with 1'* ]]
}

@test "${tag}: Debian Buster: needed, with --force" {
    scope standard
    # commands that may need it, they do, --force, success
    run ch-image -v build --force -t fakeroot-temp -f - . <<'EOF'
FROM debian:buster
RUN apt-get update && apt-get install -y openssh-client
EOF
    echo "$output"
    [[ $status -eq 0 ]]
    [[ $output = *'will use --force'* ]]
    [[ $output = *'--force: init OK & modified 1 RUN instructions'* ]]
}
