// Copyright (c) 1999,2004
// Utrecht University (The Netherlands),
// ETH Zurich (Switzerland),
// INRIA Sophia-Antipolis (France),
// Max-Planck-Institute Saarbruecken (Germany),
// and Tel-Aviv University (Israel).  All rights reserved.
//
// This file is part of CGAL (www.cgal.org)
//
// $URL: https://github.com/CGAL/cgal/blob/v6.0.1/Kernel_23/include/CGAL/Bbox_2.h $
// $Id: include/CGAL/Bbox_2.h 50cfbde3b84 $
// SPDX-License-Identifier: LGPL-3.0-or-later OR LicenseRef-Commercial
//
// Author(s)     : Andreas Fabri

#ifndef CGAL_BBOX_2_H
#define CGAL_BBOX_2_H

#include <CGAL/config.h>
#include <CGAL/kernel_assertions.h>
#include <CGAL/IO/io.h>
#include <CGAL/Dimension.h>
#include <CGAL/array.h>
#include <boost/math/special_functions/next.hpp>

namespace CGAL {

template < typename T >
struct Simple_cartesian;

class Bbox_2
{
  typedef std::array<double, 4>            BBox_rep_2;

  BBox_rep_2 rep;

public:

  typedef Dimension_tag<2>  Ambient_dimension;
  typedef Dimension_tag<2>  Feature_dimension;

  typedef Simple_cartesian<double>  R;

  Bbox_2()
    : rep(CGAL::make_array(std::numeric_limits<double>::infinity(),
                           std::numeric_limits<double>::infinity(),
                           - std::numeric_limits<double>::infinity(),
                           - std::numeric_limits<double>::infinity() ))
  {}

  Bbox_2(double x_min, double y_min,
         double x_max, double y_max)
    : rep(CGAL::make_array(x_min, y_min, x_max, y_max))
{}

  inline bool       operator==(const Bbox_2 &b) const;
  inline bool       operator!=(const Bbox_2 &b) const;

  inline int        dimension() const;
  inline double     xmin() const;
  inline double     ymin() const;
  inline double     xmax() const;
  inline double     ymax() const;
  inline double x_span() const;
  inline double y_span() const;

  inline double     max BOOST_PREVENT_MACRO_SUBSTITUTION (int i) const;
  inline double     min BOOST_PREVENT_MACRO_SUBSTITUTION (int i) const;

  inline Bbox_2     operator+(const Bbox_2 &b) const;
  inline Bbox_2&     operator+=(const Bbox_2 &b);

  inline void dilate(int dist);
  inline void scale(double factor);
};

inline
double
Bbox_2::xmin() const
{ return rep[0]; }

inline
double
Bbox_2::ymin() const
{ return rep[1]; }

inline
double
Bbox_2::xmax() const
{ return rep[2]; }

inline
double
Bbox_2::ymax() const
{ return rep[3]; }

inline double Bbox_2::x_span() const {
  return xmax() - xmin();
}

inline double Bbox_2::y_span() const {
  return ymax() - ymin();
}

inline
bool
Bbox_2::operator==(const Bbox_2 &b) const
{
  return    xmin() == b.xmin() && xmax() == b.xmax()
         && ymin() == b.ymin() && ymax() == b.ymax();
}

inline
bool
Bbox_2::operator!=(const Bbox_2 &b) const
{
  return ! (b == *this);
}

inline
int
Bbox_2::dimension() const
{ return 2; }

inline
double
Bbox_2::min BOOST_PREVENT_MACRO_SUBSTITUTION (int i) const
{
  CGAL_kernel_precondition( (i == 0 ) || ( i == 1 ) );
  if(i == 0) { return xmin(); }
  return ymin();
}

inline
double
Bbox_2::max BOOST_PREVENT_MACRO_SUBSTITUTION (int i) const
{
  CGAL_kernel_precondition( (i == 0 ) || ( i == 1 ) );
  if(i == 0) { return xmax(); }
  return ymax();
}

inline
Bbox_2
Bbox_2::operator+(const Bbox_2 &b) const
{
  return Bbox_2((std::min)(xmin(), b.xmin()),
                (std::min)(ymin(), b.ymin()),
                (std::max)(xmax(), b.xmax()),
                (std::max)(ymax(), b.ymax()));
}

inline
Bbox_2&
Bbox_2::operator+=(const Bbox_2& b)
{
  rep[0] = (std::min)(xmin(), b.xmin());
  rep[1] = (std::min)(ymin(), b.ymin());
  rep[2] = (std::max)(xmax(), b.xmax());
  rep[3] = (std::max)(ymax(), b.ymax());
  return *this;
}
inline
void
Bbox_2::dilate(int dist)
{
  using boost::math::float_advance;
  rep[0] = float_advance(rep[0],-dist);
  rep[1] = float_advance(rep[1],-dist);
  rep[2] = float_advance(rep[2],dist);
  rep[3] = float_advance(rep[3],dist);
}

inline
void
Bbox_2::scale(double factor)
{
  CGAL_precondition(factor > 0);

  if(factor == 1.)
    return;

  std::array<double, 2> half_width = { (xmax() - xmin()) * 0.5,
                                       (ymax() - ymin()) * 0.5 };
  std::array<double, 2> center = { xmin() + half_width[0],
                                   ymin() + half_width[1] };
  rep[0] = center[0] - factor * half_width[0];
  rep[1] = center[1] - factor * half_width[1];
  rep[2] = center[0] + factor * half_width[0];
  rep[3] = center[1] + factor * half_width[1];
}

inline
bool
do_overlap(const Bbox_2 &bb1, const Bbox_2 &bb2)
{
    // check for emptiness ??
    if (bb1.xmax() < bb2.xmin() || bb2.xmax() < bb1.xmin())
        return false;
    if (bb1.ymax() < bb2.ymin() || bb2.ymax() < bb1.ymin())
        return false;
    return true;
}

inline
std::ostream&
operator<<(std::ostream &os, const Bbox_2 &b)
{
    switch(IO::get_mode(os)) {
    case IO::ASCII :
        os << b.xmin() << ' ' << b.ymin() << ' '
           << b.xmax() << ' ' << b.ymax();
        break;
    case IO::BINARY :
        write(os, b.xmin());
        write(os, b.ymin());
        write(os, b.xmax());
        write(os, b.ymax());
        break;
    case IO::PRETTY :
    default:
        os << "Bbox_2(" << b.xmin() << ", " << b.ymin() << ", "
                        << b.xmax() << ", " << b.ymax() << ")";
        break;
    }
    return os;
}

inline
std::istream&
operator>>(std::istream &is, Bbox_2 &b)
{
    double xmin = 0;
    double ymin = 0;
    double xmax = 0;
    double ymax = 0;

    switch(IO::get_mode(is)) {
    case IO::ASCII :
        is >> IO::iformat(xmin) >> IO::iformat(ymin) >> IO::iformat(xmax) >> IO::iformat(ymax);
        break;
    case IO::BINARY :
        read(is, xmin);
        read(is, ymin);
        read(is, xmax);
        read(is, ymax);
        break;
    case IO::PRETTY :
        break;
    }
    if (is)
      b = Bbox_2(xmin, ymin, xmax, ymax);
    return is;
}

template <class Input_iterator, class Traits>
Bbox_2 bbox_2(Input_iterator begin, Input_iterator end, const Traits& traits)
{
  if (begin==end) return Bbox_2();
  typename Traits::Construct_bbox_2 get_bbox = traits.construct_bbox_2_object();
  Bbox_2 res = get_bbox( *begin );
  for (++begin; begin!=end; ++begin)
    res += get_bbox( *begin );
  return res;
}

template <class Input_iterator>
Bbox_2 bbox_2(Input_iterator begin, Input_iterator end)
{
  if (begin==end) return Bbox_2();
  Bbox_2 res = begin->bbox();
  for (++begin; begin!=end; ++begin)
    res += begin->bbox();
  return res;
}

} //namespace CGAL

#endif // CGAL_BBOX_2_H
