﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class AssociateTargetsWithJobRequest : public IoTRequest {
 public:
  AWS_IOT_API AssociateTargetsWithJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateTargetsWithJob"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>A list of thing group ARNs that define the targets of the job.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTargets() const { return m_targets; }
  inline bool TargetsHasBeenSet() const { return m_targetsHasBeenSet; }
  template <typename TargetsT = Aws::Vector<Aws::String>>
  void SetTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets = std::forward<TargetsT>(value);
  }
  template <typename TargetsT = Aws::Vector<Aws::String>>
  AssociateTargetsWithJobRequest& WithTargets(TargetsT&& value) {
    SetTargets(std::forward<TargetsT>(value));
    return *this;
  }
  template <typename TargetsT = Aws::String>
  AssociateTargetsWithJobRequest& AddTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets.emplace_back(std::forward<TargetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier you assigned to this job when it was created.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  AssociateTargetsWithJobRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional comment string describing why the job was associated with the
   * targets.</p>
   */
  inline const Aws::String& GetComment() const { return m_comment; }
  inline bool CommentHasBeenSet() const { return m_commentHasBeenSet; }
  template <typename CommentT = Aws::String>
  void SetComment(CommentT&& value) {
    m_commentHasBeenSet = true;
    m_comment = std::forward<CommentT>(value);
  }
  template <typename CommentT = Aws::String>
  AssociateTargetsWithJobRequest& WithComment(CommentT&& value) {
    SetComment(std::forward<CommentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace used to indicate that a job is a customer-managed job.</p>
   * <p>When you specify a value for this parameter, Amazon Web Services IoT Core
   * sends jobs notifications to MQTT topics that contain the value in the following
   * format.</p> <p>
   * <code>$aws/things/<i>THING_NAME</i>/jobs/<i>JOB_ID</i>/notify-namespace-<i>NAMESPACE_ID</i>/</code>
   * </p>  <p>The <code>namespaceId</code> feature is only supported by IoT
   * Greengrass at this time. For more information, see <a
   * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/setting-up.html">Setting
   * up IoT Greengrass core devices.</a> </p>
   */
  inline const Aws::String& GetNamespaceId() const { return m_namespaceId; }
  inline bool NamespaceIdHasBeenSet() const { return m_namespaceIdHasBeenSet; }
  template <typename NamespaceIdT = Aws::String>
  void SetNamespaceId(NamespaceIdT&& value) {
    m_namespaceIdHasBeenSet = true;
    m_namespaceId = std::forward<NamespaceIdT>(value);
  }
  template <typename NamespaceIdT = Aws::String>
  AssociateTargetsWithJobRequest& WithNamespaceId(NamespaceIdT&& value) {
    SetNamespaceId(std::forward<NamespaceIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_targets;
  bool m_targetsHasBeenSet = false;

  Aws::String m_jobId;
  bool m_jobIdHasBeenSet = false;

  Aws::String m_comment;
  bool m_commentHasBeenSet = false;

  Aws::String m_namespaceId;
  bool m_namespaceIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
